/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package org.jab.docsearch;

import java.util.Locale;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jab.docsearch.constants.OSType;
import org.jab.docsearch.utils.Logging;

/**
 * This class contains all variables for a running session.<br>
 * The class is a Singleton.
 */
public final class Environment {
    /**
     * Log4J
     */
    private final Logger logger = LogManager.getLogger(getClass().getName());
    /**
     * Environment variable
     */
    private static final Environment env = new Environment();
    /**
     * Is GUI mode
     */
    private boolean isGUIMode;
    /**
     * OS Type
     * @see OSType
     */
    private int oSType;
    /**
     * CD directory
     */
    private String cDROMDir;
    /**
     * DocSearcher runs in Webstart?
     */
    private boolean isWebStart;

    /**
     * Default constructor
     */
    private Environment() {
        logger.info("Environment() created");

        // checking for Java Web Start
        try {
          Class.forName("javax.jnlp.ServiceManager");
          isWebStart = true;
        }
        catch (ClassNotFoundException cnfe) {
            // ignore
        }
        logger.log(Logging.NOTICE, "Environment() Java Web Start - {}", isWebStart);

        // Locale
        localeInformation();
    }


    /**
     * Gets instance of Environment
     *
     * @return  instance of Environment
     */
    public static Environment getInstance() {
        return env;
    }


    /**
     * Is GUI mode
     *
     * @return  true is GUI mode on
     */
    public boolean isGUIMode() {
       return isGUIMode;
    }


    /**
     * Sets GUI mode
     *
     * @param isGUIMode  GUI mode
     */
    public void setGUIMode(boolean isGUIMode) {
       this.isGUIMode = isGUIMode;
    }


    /**
     * Gets OS Type
     *
     * @see  OSType
     * @return  OS type
     */
    public int getOSType() {
        return oSType;
    }


    /**
     * Sets OS Type
     *
     * @see  OSType
     * @param oSType  OS Type
     */
    public void setOSType(int oSType) {
        this.oSType = oSType;
    }


    /**
     * Gets CDROM directory
     *
     * @return  CDROM directory
     */
    public String getCDROMDir() {
        return cDROMDir;
    }


    /**
     * Sets CDROM directory
     *
     * @param cDROMDir  CDROM directory
     */
    public void setCDROMDir(String cDROMDir) {
        this.cDROMDir = cDROMDir;
    }


    /**
     * Run DocSearcher in Web Start?
     *
     * @return  true, if DocSearcher runs in Web Start
     */
    public boolean isWebStart() {
        return isWebStart;
    }


    /**
     * Method shows the locale information.
     */
    private void localeInformation() {
        // Get current Locale information
        Locale locale = Locale.getDefault();
        String language = locale.getLanguage();
        String country = locale.getCountry();
        String variant = locale.getVariant();

        if (!country.isEmpty()) {
            country = "_".concat(country);
        }
        if (!variant.isEmpty()) {
            variant = "_".concat(variant);
        }

        String locality = language + country + variant;
        logger.log(Logging.NOTICE, "localeInformation() using locale: {}", locality);
    }
}
